<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class VolunteerController extends Controller
{
    /**
     * Display the volunteers list.
     */
    public function index(Request $request): Response
    {
        $volunteers = User::where('user_type_id', 4)->paginate(config('app.pagination'));

        return Inertia::render('Volunteer/Index', [
            'volunteers' => $volunteers,
            'success_msg' => Session::has('success_msg') ? Session::get('success_msg') : null,
            'error_msg' => Session::has('error_msg') ? Session::get('error_msg') : null
        ]);
    }

    /**
     * Display the volunteer create form.
     */
    public function create(Request $request): Response
    {
        return Inertia::render('Volunteer/Create');
    }

    /**
     * Display the volunteer edit form.
     */
    public function edit(Request $request, $id): Response
    {
        $volunteer = User::find($id);

        return Inertia::render('Volunteer/Update', [
            'volunteer' => $volunteer
        ]);
    }

    /**
     * Delete the volunteer .
     */
    public function delete(Request $request, $id)
    {
        $volunteer = User::find($id);

        if ($volunteer->appointments->count() > 0) {
            return to_route('volunteer.index')->with('error_msg', 'Volunteer cannot be deleted, because this user has appointments');
        }

        return DB::transaction(function () use ($volunteer) {

            $volunteer->delete();
    
            return to_route('volunteer.index')->with('success_msg', 'Volunteer has been deleted successfully');
        });

    }

    /**
     * Save new volunteer.
     */
    public function save(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:100'],
            'contact' => ['required', 'numeric', 'unique:users,contact'],
            'email' => ['required', 'email', 'unique:users,email'],
            'password' => ['required', 'string', 'min:8', 'confirmed']
        ]);

        $user_params = Arr::except( $request->all(), ['password_confirmation']);
        $user_params['password'] = Hash::make($user_params['password']);
        $user_params['user_type_id'] = 4;

        return DB::transaction(function () use ($user_params) {
            $user = User::create($user_params);

            return to_route('volunteer.index')->with('success_msg', 'Volunteer has been created successfully');
        });
    }

    /**
     * Update new user.
     */
    public function update(Request $request)
    {
        $data = $request->validate([
            'id' => ['required', 'numeric'],
            'name' => ['required', 'string', 'max:100'],
            'contact' => ['required', 'numeric', 'unique:users,contact,'.$request->id],
            'email' => ['required', 'email', 'unique:users,email,'.$request->id],
        ]);

        $volunteer = User::find($request->id);
        $volunteer_params = $request->all();


        return DB::transaction(function () use ($volunteer_params, $volunteer) {
            $volunteer->update($volunteer_params);
            return to_route('volunteer.index')->with('success_msg', 'Volunteer has been updated successfully');
        });
    }
}
