<?php

namespace App\Http\Controllers;

use App\Models\Organization;
use App\Models\Sector;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Inertia\Inertia;
use Inertia\Response;

class OrganizationController extends Controller
{
    /**
     * Display the organizations list.
     */
    public function index(Request $request): Response
    {
        $organizations = Organization::with('sector')->paginate(config('app.pagination'));

        return Inertia::render('Organization/Index', [
            'organizations' => $organizations,
            'success_msg' => Session::has('success_msg') ? Session::get('success_msg') : null,
            'error_msg' => Session::has('error_msg') ? Session::get('error_msg') : null
        ]);
    }

    /**
     * Display the organization create form.
     */
    public function create(Request $request): Response
    {
        $sectors = Sector::get();

        return Inertia::render('Organization/Create',[
            'sectors' => $sectors
        ]);
    }

    /**
     * Display the organization edit form.
     */
    public function edit(Request $request, $id): Response
    {
        $organization = Organization::find($id);
        $sectors = Sector::get();

        return Inertia::render('Organization/Update', [
            'organization' => $organization,
            'sectors' => $sectors
        ]);
    }

    /**
     * Delete the organization .
     */
    public function delete(Request $request, $id)
    {
        $organization = Organization::find($id);

        if ($organization->appointments->count() > 0) {
            return to_route('organization.index')->with('error_msg', 'Organization cannot be deleted, because this organization has appointments');
        }

        $organization->delete();

        return to_route('organization.index')->with('success_msg', 'Organization has been deleted successfully');
    }

    /**
     * Save new organization.
     */
    public function save(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:100', 'unique:organizations,name'],
            'address' => ['sometimes', 'nullable', 'string', 'max:200'],
            'contact' => ['sometimes', 'nullable', 'numeric'],
            'sector_id' => ['required', 'numeric', 'exists:sectors,id'],
        ]);

        Organization::create($request->all());

        return to_route('organization.index')->with('success_msg', 'Organization has been created successfully');
    }

    /**
     * Update new organization.
     */
    public function update(Request $request)
    {
        $data = $request->validate([
            'id' => ['required', 'numeric'],
            'name' => ['required', 'string', 'max:100', 'unique:organizations,name,'.$request->id],
            'address' => ['sometimes', 'nullable', 'string', 'max:200'],
            'contact' => ['sometimes', 'nullable', 'numeric'],
            'sector_id' => ['required', 'numeric', 'exists:sectors,id'],
        ]);

        $organization = Organization::find($request->id);

        $organization->update($request->all());

        return to_route('organization.index')->with('success_msg', 'Organization has been updated successfully');
    }
}
