<?php

namespace App\Exports;

use App\Models\Appointment;
use Carbon\Carbon;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\Exportable;

class AppointmentExport implements FromCollection, WithMapping, WithHeadings
{
    use Exportable;
    protected $filters;

    public function __construct(array $filters)
    {
        $this->filters = $filters;
    }

    /**
    * @return \Illuminate\Support\Collection
    */
    public function collection()
    {
        $appointments = Appointment::with(['sector', 'createdBy']);

        if (!empty($this->filters['organization'])) {
            $appointments->where('organization', $this->filters['organization']);
        }

        if (!empty($this->filters['sector_id'])) {
            $appointments->where('sector_id', $this->filters['sector_id']);
        }

        if (!empty($this->filters['volunteer'])) {
            $appointments->where('volunteer', 'LIKE', '%'.$this->filters['volunteer'].'%');
        }

        if (!empty($this->filters['professional'])) {
            $appointments->where('professional', $this->filters['professional']);
        }

        if (!empty($this->filters['appointment_date'])) {
            $appointments->whereDate('appointment_date', $this->filters['appointment_date']);
        }

        if (!empty($this->filters['appointment_date_from'])) {
            $appointments->whereDate('appointment_date', '>=', $this->filters['appointment_date_from']);
        }

        if (!empty($this->filters['appointment_date_to'])) {
            $appointments->whereDate('appointment_date', '<=', $this->filters['appointment_date_to']);
        }
        
        $appointments = $appointments->get();

        return $appointments;
    }

    public function map($appointment): array
    {
        return [
            $appointment->title,
            $appointment->description,
            $appointment->sector->name,
            $appointment->organization,
            $appointment->volunteer,
            $appointment->createdBy->name,
            $appointment->professional,
            $appointment->professional_person_email,
            $appointment->client,
            $appointment->no_of_meeting,
            Carbon::parse($appointment->appointment_date)->format('d-m-Y'),
            $appointment->appointment_start_time,
            $appointment->appointment_end_time,
        ];
    }

    public function headings(): array
    {
        return [
            'Title',
            'Description',
            'Sector',
            'Organization',
            'Volunteer',
            'User',
            'Professional Person',
            'Professional Person Email Address',
            'Client',
            'No. Of Meeting',
            'Appointment Date',
            'Appointment Start Time',
            'Appointment End Time',
        ];
    }
}
